
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
      *  \brief This file includes the DES ATP tests
   *
   *  \version CRYS_SEP_ATP_DES_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_DES.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern CRYSError_t             TST_ATP_Error;

CRYS_DESUserContext_t	DESContextID;


/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   char        *FailureFunctionNameOut_ptr);


/********************************************************************************
* Function Name:
*  SEPQA_DES_NotIntegrated
*
* Inputs:
* IV_ptr - this parameter is the buffer of the IV or counters on mode CTR.
*                          In ECB mode this parameter has no use.
*                          In CBC and MAC modes this parameter should containe the IV values.
*                          in CTR mode this value shuld contain the init counter.
*
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function is qa external application on the SEP, 
* that executes AES not integrated test
* Algorithm:     
* 1. Allocate memory buffers
* 2. Calling the DES init function
* 3. Calling the DES Block function
* 4. Calling the Free function
* 5. Compare the result to expected result
* 6. Free allocated buffers
********************************************************************************/
CRYSError_t SEPQA_DES_NotIntegrated(CRYS_DES_Iv_t             IV_ptr,                 /* in */ 
									CRYS_DES_Key_t           *Key_ptr,                /* in */ 
									CRYS_DES_NumOfKeys_t      NumOfKeys,              /* in */ 
									CRYS_DES_EncryptMode_t    EncryptDecryptFlag,     /* in */  
									CRYS_DES_OperationMode_t  OperationMode,          /* in */     
									DxUint8_t                *DataIn_ptr,             /* in */ 
									DxUint32_t                DataInSize,             /* in */ 
									DxUint8_t                *ExpectedDataOut_ptr,        
									DxUint32_t                DataExpSize,
									DxUint32_t				  OutputLocation,
									char                     *FailureFunctionNameOut_ptr)									
{
	/********* Variables Definitions ************/
	CRYSError_t				TST_TestStatus = TESTPASS;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;
	DxUint32_t               Block_index = 0;
	DxUint32_t               NumOfBlocks = 0;

	/* data expected size = data input size */
	DataExpSize = DataInSize;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/* 2. calling the DES init function */ 
	/************************************/ 
	TST_ATP_Error = CRYS_DES_Init(&DESContextID,
		IV_ptr, 
		Key_ptr,
		NumOfKeys, 
		EncryptDecryptFlag, 
		OperationMode);                          

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_DES_Init",
												FailureFunctionNameOut_ptr);

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 3. calling the DES Block function */
	/*************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS;


	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*DES block loop*/ 
		{   
			if (EncryptDecryptFlag == CRYS_DES_Encrypt)
			{ 
				TST_ATP_Error = CRYS_DES_Block( &DESContextID,     
					pDataIn+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index),         
					TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
					pDataOut+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index));

			}
			else
			{                                       
				TST_ATP_Error = CRYS_DES_Block( &DESContextID,     
					ExpectedDataOut_ptr+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index),        
					TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
					pDataOut+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index)); 
			}

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_DES_Block",
				FailureFunctionNameOut_ptr);
				
			if (TST_TestStatus != TESTPASS) 
				goto End;
		}/*End DES block loop*/ 
	}
	else /* input buffer = output buffer */ 
	{
		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{
			if (EncryptDecryptFlag == CRYS_DES_Encrypt)
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index),
					pDataIn+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index),
					TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS);  
			}
			else
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index),
					ExpectedDataOut_ptr+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index), 
					TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS);  
			}
		}           

		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{   
			TST_ATP_Error = CRYS_DES_Block(&DESContextID,           
				pDataOut+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index),        
				TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
				pDataOut+(TST_MAX_SINGLE_DES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

			if (EncryptDecryptFlag == CRYS_DES_Encrypt)
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_DES_Block - Encrypt",
				FailureFunctionNameOut_ptr);
				

			else 
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_DES_Block - Decrypt",
				FailureFunctionNameOut_ptr);       
				

			if (TST_TestStatus != TESTPASS) 
				goto End;
		}

	}

	/* 4. calling the Free function */ 
	/********************************/     
	TST_ATP_Error = CRYS_DES_Free(&DESContextID);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_DES_Free",
		FailureFunctionNameOut_ptr);       
		

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Compare the result to expected result */ 
	/********************************************/ 
	if (EncryptDecryptFlag == CRYS_DES_Encrypt)
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
			ExpectedDataOut_ptr,
			DataInSize);
	}
	else
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
			pDataIn,
			DataInSize);                
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);
		

End:
	/* 6. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;
}



/********************************************************************************
* Function Name:
*  SEPQA_DES_Integrated
*
* Inputs:
* IV_ptr - this parameter is the buffer of the IV or counters on mode CTR.
*                          In ECB mode this parameter has no use.
*                          In CBC and MAC modes this parameter should containe the IV values.
*                          in CTR mode this value shuld contain the init counter.
*
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function is qa external application on the SEP, 
* that executes AES not integrated test
* Algorithm:     
* 1. Allocate memory buffers
* 2. Calling the DES function
* 4. Calling the Free function
* 5. Compare the result to expected result
* 6. Free allocated buffers
********************************************************************************/
CRYSError_t SEPQA_DES_Integrated(
								 CRYS_DES_Iv_t             IV_ptr,                 /* in */ 
								 CRYS_DES_Key_t           *Key_ptr,                /* in */ 
								 CRYS_DES_NumOfKeys_t      NumOfKeys,              /* in */ 
								 CRYS_DES_EncryptMode_t    EncryptDecryptFlag,     /* in */  
								 CRYS_DES_OperationMode_t  OperationMode,          /* in */     
								 DxUint8_t                *DataIn_ptr,             /* in */ 
								 DxUint32_t                DataInSize,             /* in */ 
								 DxUint8_t                *ExpectedData_ptr,        
								 DxUint32_t                DataExpSize,
								 DxUint32_t				   OutputLocation,
								 char                     *FailureFunctionNameOut_ptr)
								 
{
	/********* TST Variables Definitions ************/
	CRYSError_t				TST_TestStatus=0;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;

	/* data expected size = data input size */
	DataExpSize = DataInSize;
	

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/* 2. calling the DES integrated function */
	/******************************************/ 
	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		if (EncryptDecryptFlag ==CRYS_DES_Encrypt) 
		{
			
			TST_ATP_Error = CRYS_DES(
									IV_ptr,
									Key_ptr,
									NumOfKeys,
									EncryptDecryptFlag,
									OperationMode,
									pDataIn, 
									DataInSize, 
									pDataOut);
									

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_DES - Encrypt",
														FailureFunctionNameOut_ptr);
																
		}
		else
		{
			TST_ATP_Error = CRYS_DES(IV_ptr,
									Key_ptr,
									NumOfKeys,
									EncryptDecryptFlag,
									OperationMode,
									ExpectedData_ptr, 
									DataInSize, 
									pDataOut);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_DES - Decrypt",
														FailureFunctionNameOut_ptr);
														
			
		}

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}
	else /* input buffer = output buffer */ 
	{
		if (EncryptDecryptFlag == CRYS_DES_Encrypt)
		{
			/*Copying the input buffer onto output buffer not to overwrite test data*/
			DX_VOS_FastMemCpy(pDataOut,
				pDataIn,
				DataInSize);  
		}
		else
		{
			/*Copying the input buffer onto output buffer not to overwrite test data*/
			DX_VOS_FastMemCpy(pDataOut,
				ExpectedData_ptr, 
				DataInSize);  
		}
		
		TST_ATP_Error = CRYS_DES(IV_ptr,
						Key_ptr,
						NumOfKeys,
						EncryptDecryptFlag,
						OperationMode,
						pDataOut, 
						DataInSize, 
						pDataOut);

		if (EncryptDecryptFlag == CRYS_DES_Encrypt)
			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_DES Encrypt",
														FailureFunctionNameOut_ptr);
														
		else
			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
														"CRYS_DES Decrypt",
														FailureFunctionNameOut_ptr);
														

		if (TST_TestStatus != TESTPASS) 
			goto End;
		            
	}   

	/* Compare the result to expected result */ 
	/*****************************************/                                                             
	if (EncryptDecryptFlag == CRYS_DES_Encrypt)  
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
			pDataOut,
			ExpectedData_ptr,
			DataInSize);
	}
	else
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
			pDataOut,
			pDataIn,
			DataInSize);                
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);       
		

End:
	/* 6. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;
}







